/***************************************************************************
This is a program to approximate a cubically symmetric periodic solution to
Laplacian u = (u>0) ? -1 : 1
and compute the absolute value of its gradient at two different points
where u=0.
****************************************************************************
Programmed by Luis Silvestre in January 2007.
***************************************************************************/


#include <iostream>
#include <fstream>
#include <sstream>
#include <cmath>
#include <cfloat>
#include "scube.h"

using namespace std;

#define Pi 3.141592653589793238

inline double sqr(double x){
  return x*x;
}

void init(scube& c){
  c.clear();
  int n = c.size();
  for(int x=0; x<n; ++x){
    for(int y=0; y<=x; ++y){
      for(int z=0; z<=y; ++z){
        c.get(x,y,z)=cos(Pi * x / (n-1)) + cos(Pi * y /(n-1)) + cos(Pi * z /(n-1));
      }
    }
  }
}

void renormalize(scube& c){
  double s=0;
  int n = c.size();
  for(int x=0; x<n; ++x){
    for(int y=0; y<=x; ++y){
      for(int z=0; z<=y; ++z){
        s += c.plainget(x,y,z);
      }
    }
  }
  s /= c.raw().size();
  for(int x=0; x<n; ++x){
    for(int y=0; y<=x; ++y){
      for(int z=0; z<=y; ++z){
        c.plainget(x,y,z) -= s;
      }
    }
  }
}

double iterate(scube& c){
  int n = c.size();
  scube nc(n);
  double d=0;
  double h = 1.0 / (n-1);
  for(int x=0; x<n; ++x){
    for(int y=0; y<=x; ++y){
      for(int z=0; z<=y; ++z){
        double au=0.;
        double cc= c.plainget(x,y,z);
        if(cc > 0.1*h*h*h) au=1;
        if(cc < -0.1*h*h*h) au=-1;
        nc.plainget(x,y,z) = cc/2 + (c.get(x+1,y,z)+c.get(x-1,y,z)    // This computation should make 
            +c.get(x,y+1,z)+c.get(x,y-1,z)    // the laplacian equal to -au
                +c.get(x,y,z+1)+c.get(x,y,z-1)
                +h*h*au)/12;
        d >?= abs(nc.plainget(x,y,z)-cc);
      }
    }
  }
  c = nc;
  return d;
}

void evolve(scube& c){
  int n = c.size();
  double h = 1.0 / (n-1);
  double e = 0.00000001 * h * h;
  cout << "Target error: " << e << endl;
  double esf = 1;
  for(int i=0; esf>e; esf=iterate(c)){
    if (i%50 == 0) cout << "Gridsize: "<< n << ", after " << i << " its., the error is aprox. " << esf << " and u(1,.5,0) = " << c.get(n-1,n/2,0) << endl;
    ++i;
    if(i%20 == 0) renormalize(c);
  }
  cout << "Finally, the error is aproximately " << iterate(c) << " and u(1,.5,0) = " << c.get(n-1,n/2,0) << endl;
}

/*void sliceforgnuplot(scube& c, const char* fn){
  ofstream ofs(fn);
  for(int i=-c.size()+1; i<c.size(); ++i){
    for(int j=-c.size()+1; j<c.size(); ++j){
      ofs << i << " " << j << " " << c.get(0,i,j) << endl;
    }
    ofs << endl;
  }
  ofs << endl;
  for(int i=-c.size()+1; i<c.size(); ++i){
    for(int j=-c.size()+1; j<c.size(); ++j){
      ofs << i << " " << j << " " << c.get(c.size(),i,j) << endl;
    }
    ofs << endl;
  }
}*/

void report(scube& c, const char* fn){
  ofstream ofs(fn);
  
  int n = c.size();
  double h = 1.0 / (n-1);
  ofs << "**************************************************************************" << endl;
  ofs << "*****************Report on the results for size " << n << "************************" << endl;
  ofs << "**************************************************************************" << endl;
  ofs << "Something was computed " << endl;
  ofs << "At (0,0,0), we have " << c.plainget(0,0,0);
  
  double l = (-6*c.get(0,0,0)+(c.get(1,0,0)+c.get(-1,0,0)+c.get(0,1,0)+c.get(0,-1,0)+c.get(0,0,1)+c.get(0,0,-1)))/sqr(h);
  ofs << " and the Laplacian is " << l << endl;
  
  ofs << "Now I'll compute the derivative of u at some points" << endl;
  
  double dx = (c.get(n/2+1,n/2,n/2)-c.get(n/2-1,n/2,n/2))/(2*h);
  double dy = (c.get(n/2,n/2+1,n/2)-c.get(n/2,n/2-1,n/2))/(2*h);
  double dz = (c.get(n/2,n/2,n/2+1)-c.get(n/2,n/2,n/2-1))/(2*h);
  ofs << "**************************************************************************" << endl;
  ofs << "At (1/2,1/2,1/2), u = " << c.get(n/2,n/2,n/2) << " and the gradient is: " << dx << ", " << dy << ", " << dz << endl;
  ofs << "which has norm " << sqrt(dx*dx+dy*dy+dz*dz) << endl;
  ofs << "But a more accurate approximation should be " << (c.get(n/2,n/2,n/2)-c.get(n/2+1,n/2+1,n/2+1)+3*h*h/2)/(sqrt(3)*h) << endl;
  ofs << "**************************************************************************" << endl;


  dx = (c.get(n,n/2,0)-c.get(n-2,n/2,0))/(2*h);
  dy = (c.get(n-1,n/2+1,0)-c.get(n-1,n/2-1,0))/(2*h);
  dz = (c.get(n-1,n/2,1)-c.get(n-1,n/2,-1))/(2*h);
  ofs << "At (1,1/2,0), u =" << c.get(n-1,n/2,0) << " and the gradient is: " << dx << ", " << dy << ", " << dz << endl;
  ofs << "which has norm " << sqrt(dx*dx+dy*dy+dz*dz) << endl;
  ofs << "A better approximation of u_\\nu should be " << (-c.get(n-1,n/2+1,0)+c.get(n-1,n/2,0)+h*h/2)/h << endl;
  ofs << "**************************************************************************" << endl;
  ofs << "**************************************************************************" << endl;
  ofs << endl << "I hope this was useful." << endl;
  ofs << "**************************************************************************" << endl;
}

int main(int argc, char *argv[])
{
  int n;
  cout << "**************************************************************************" << endl;
  cout << "This will be a solver for minimizers of the functional" << endl;
  cout << " ---> J(u) = \\int |Du|^2 - |u| dx " << endl;
  cout << "**************************************************************************" << endl;
  cout << "So you know, sizeof(double)= " << sizeof(double) << ", sizeof(long double)=" << sizeof(long double) << ". I am using (double)." << endl;

  double dp=1;
  int k = 200;
  while(dp==1){
    dp = dp + pow((double)0.1,k--);
  }
  cout << " The arithmetic presition in this machine is " << dp-1 << endl;


  cout << endl << "How many grid points should I use? (recommended odd):";
  cin >> n;
  // n=11;
  
  cout << "After symmetry, the cube is actually " << 2*n << "x" << 2*n << "x" << 2*n << endl;
  cout << "Internally, this program only computes the values of u in the tetrahedron 0<= x < " << n << ", 0 <=y<=x, 0<=z<=y" << endl;
  cout << "since this is the fundamental domain of the cube after all the cubic symmetries." << endl;
  
  int m = 7;
  scube c(m);
  cout << "Initializing. " << endl;
  
  init(c);
  
  cout << "Starting computation. " << endl;
  
  while(true){
    evolve(c);
    
    char *fn,*fn2;
    fn = new char[35];
    fn2 = new char[35];
    if(argc==1){
      stringstream sfn;
      sfn << "report";
      sfn << m;
      strcpy(fn,sfn.str().c_str());
      cout << "Report file name: " << fn << endl;
      
      stringstream sfn2;
    }
    else fn = argv[1];
    
    cout << "Writing Report." << endl;
    report(c,fn);
  
/*    if(argc<3){
      stringstream sfn2;
      sfn2 << "plot";
      sfn2 << m;
      strcpy(fn2,sfn2.str().c_str());
      cout << "GNU Plot file name: " << fn2 << endl;
    }  else fn2 = argv[2];
  
    cout << "Writing Plot file." << endl;
    sliceforgnuplot(c,fn2);*/
    
    string s = "cat ";
    s += fn;
    system(s.c_str());
    
    delete fn;
    delete fn2;
    if(m==n) return EXIT_SUCCESS;
    m = 3*m / 2;
    m = (m%2 == 0) ? m+1 : m;
    m <?= n;
    c = c.interpolate(m);
  }
}
