/************************************************************
Program to compute the effective conductivity of a composite
using an adaptive finite element method.

Programmed by: Luca Heltai - Feb 2007

Requires the library dealii to compile.
*************************************************************/

#include <base/quadrature_lib.h>
#include <base/function.h>
#include <base/logstream.h>
#include <base/parameter_handler.h>
#include <lac/vector.h>
#include <lac/compressed_sparsity_pattern.h>
#include <lac/full_matrix.h>
#include <lac/sparse_matrix.h>
#include <lac/solver_cg.h>
#include <lac/precondition.h>
#include <grid/tria.h>
#include <dofs/dof_handler.h>
#include <grid/grid_generator.h>
#include <grid/tria_accessor.h>
#include <grid/tria_iterator.h>
#include <grid/tria_boundary_lib.h>
#include <dofs/dof_accessor.h>
#include <dofs/dof_tools.h>
#include <fe/fe_values.h>
#include <numerics/vectors.h>
#include <numerics/matrices.h>
#include <numerics/data_out.h>
#include "parsed_function.templates.h"

#include <fstream>
#include <iostream>
 
#include <fe/fe_q.h>
#include <grid/grid_out.h>
 
 
#include <dofs/dof_constraints.h>
 
#include <grid/grid_refinement.h>
 
#include <numerics/error_estimator.h>
 
using namespace dealii;
template <int dim>
class LaplaceProblem 
{
public:
  LaplaceProblem ();
  ~LaplaceProblem ();
 
  void run ();
     
private:
  void set_prms();
  void setup_system ();
  void assemble_system ();
  void solve ();
  void refine_grid ();
  void output_results (const unsigned int cycle);
 
  Triangulation<dim>   triangulation;
 
  DoFHandler<dim>      dof_handler;
  FE_Q<dim>            fe;
 
  ConstraintMatrix     hanging_node_constraints;
 
  SparsityPattern      sparsity_pattern;
  SparseMatrix<double> system_matrix;
 
  Vector<double>       solution;
  Vector<double>       system_rhs;
  
  ParsedFunction<dim>  a_coeff;
  ParameterHandler     prm;
  
  GridOut	       grid_out;
  DataOut<dim>	       data_out;
  
  unsigned int	       global_ref;
  unsigned int	       n_cycles;
};

template <int dim>
LaplaceProblem<dim>::LaplaceProblem () :
  dof_handler (triangulation),
  fe (2)
{}
template <int dim>
LaplaceProblem<dim>::~LaplaceProblem () 
{
  dof_handler.clear ();
}
template <int dim>
void LaplaceProblem<dim>::setup_system ()
{
  dof_handler.distribute_dofs (fe);
  
  CompressedSparsityPattern compressed_pattern (dof_handler.n_dofs(),
						dof_handler.n_dofs());
  DoFTools::make_sparsity_pattern (dof_handler, compressed_pattern);

  solution.reinit (dof_handler.n_dofs());
  system_rhs.reinit (dof_handler.n_dofs());
 
   
  hanging_node_constraints.clear ();
  DoFTools::make_hanging_node_constraints (dof_handler,
                                           hanging_node_constraints);
 
  hanging_node_constraints.close ();
 
  hanging_node_constraints.condense (compressed_pattern);
 
  sparsity_pattern.copy_from(compressed_pattern);
 
  system_matrix.reinit (sparsity_pattern);
}
template <int dim>
void LaplaceProblem<dim>::assemble_system () 
{  
  const QGauss<dim>  quadrature_formula(3);
  ZeroFunction<dim> zerof;
  Vector<double> tmp(system_rhs);
  MatrixCreator::create_laplace_matrix(dof_handler, quadrature_formula,
				       system_matrix, zerof, tmp, &a_coeff);

  hanging_node_constraints.condense (system_matrix);
  hanging_node_constraints.condense (system_rhs);
 
  std::map<unsigned char,const Function<dim> *> bmap;
  ConstantFunction<dim> onef(1.);
  bmap[2*(dim-1)] = &zerof;
  bmap[2*(dim-1)+1] = &onef;
  
  
  std::map<unsigned int,double> boundary_values;
  VectorTools::interpolate_boundary_values (dof_handler,
                                            bmap,
                                            boundary_values);
  MatrixTools::apply_boundary_values (boundary_values,
                                      system_matrix,
                                      solution,
                                      system_rhs);
}
template <int dim>
void LaplaceProblem<dim>::solve () 
{
  SolverControl           solver_control (1000, 1e-12*system_rhs.l2_norm());
  SolverCG<>              cg (solver_control);
 
  PreconditionSSOR<> preconditioner;
  preconditioner.initialize(system_matrix, 1.2);
 
  cg.solve (system_matrix, solution, system_rhs,
            preconditioner);
 
  hanging_node_constraints.distribute (solution);
  
  Vector<double> difference(dof_handler.n_dofs());
  QGauss<dim> quad(3);
  VectorTools::integrate_difference(dof_handler,
				    solution,
				    ZeroFunction<dim>(),
				    difference,
				    quad,
				    VectorTools::H1_seminorm,
				    &a_coeff);
				    
  std::cout << "   Energy norm:                  "
	    << difference.l2_norm()*difference.l2_norm()
	    << std::endl;
  
}
template <int dim>
void LaplaceProblem<dim>::refine_grid ()
{
  Vector<float> estimated_error_per_cell (triangulation.n_active_cells());
 
  KellyErrorEstimator<dim>::estimate (dof_handler,
                                      QGauss<dim-1>(3),
                                      typename FunctionMap<dim>::type(),
                                      solution,
                                      estimated_error_per_cell);
 
  GridRefinement::refine_and_coarsen_fixed_number (triangulation,
                                                   estimated_error_per_cell,
                                                   0.3, 0.03);
 
  triangulation.execute_coarsening_and_refinement ();
}
template <int dim>
void LaplaceProblem<dim>::output_results (const unsigned int cycle)
{
  Assert (cycle < 10, ExcNotImplemented());

  if(grid_out.default_suffix() != "") {

    std::string filename = "grid-";
    filename += ('0' + cycle);
    filename += grid_out.default_suffix();
    
    std::ofstream output (filename.c_str());
    grid_out.write (triangulation, output);
  }
  
  if(data_out.default_suffix() != "") {
    
    data_out.clear();
    data_out.attach_dof_handler (dof_handler);
    data_out.add_data_vector (solution, "solution");
    data_out.build_patches ();
   
    std::string filename = "solution-";
    filename += ('0' + cycle);
    filename += data_out.default_suffix();
    
    std::ofstream output (filename.c_str());
    data_out.write (output);
  }
}



template <int dim>
void LaplaceProblem<dim>::set_prms () 
{
  prm.declare_entry("Global refinement", "3", Patterns::Integer());
  prm.declare_entry("Local refinement", "3", Patterns::Integer());


  prm.enter_subsection("Coefficient");
  ParsedFunction<dim>::declare_parameters(prm);
  prm.leave_subsection();

  prm.enter_subsection("Grid Out");
  GridOut::declare_parameters(prm);
  prm.leave_subsection();

  prm.enter_subsection("Data Out");
  DataOut<dim>::declare_parameters(prm);
  prm.leave_subsection();
  
  prm.read_input("coefficent.prm");

  prm.enter_subsection("Coefficient");
  a_coeff.parse_parameters(prm);
  prm.leave_subsection();

  prm.enter_subsection("Grid Out");
  grid_out.parse_parameters(prm);
  prm.leave_subsection();

  prm.enter_subsection("Data Out");
  data_out.parse_parameters(prm);
  prm.leave_subsection();

  global_ref = prm.get_integer("Global refinement");
  n_cycles   = prm.get_integer("Local refinement");
} 

template <int dim>
void LaplaceProblem<dim>::run () 
{
  set_prms();
  
  for (unsigned int cycle=0; cycle< n_cycles; ++cycle)
    {
      std::cout << "Cycle " << cycle << ':' << std::endl;
 
      if (cycle == 0)
        {
	  Point<dim> corner;
	  for(unsigned int i=0; i<dim; ++i) corner(i) = 1;
	  
          GridGenerator::hyper_rectangle (triangulation, Point<dim>(), 
					  corner, true);
          triangulation.refine_global (global_ref);
        }
      else
        refine_grid ();
       
 
      std::cout << "   Number of active cells:       "
                << triangulation.n_active_cells()
                << std::endl;
 
      setup_system ();
 
      std::cout << "   Number of degrees of freedom: "
                << dof_handler.n_dofs()
                << std::endl;
       
      assemble_system ();
      solve ();
      output_results (cycle);


	/*** Added later *******/
	Point<dim> p0(0.,1.,0.5);
	
	double value0 = VectorTools::point_value(dof_handler, solution, p0);
	
	Point<dim> p1(0.5,0.5,0.5);
	
	double value1 = VectorTools::point_value(dof_handler, solution, p1);
	std::cout  << "At (0,1,1/2), I have 0.5+" << value0-0.5 << ", and at (1/2,1/2,1/2), I have 0.5+" << value1-0.5 << std::endl;
    }
}
int main () 
{
 
  try
    {
      deallog.depth_console (0);
 
      LaplaceProblem<3> laplace_problem;
      laplace_problem.run ();
    }
  catch (std::exception &exc)
    {
      std::cerr << std::endl << std::endl
                << "----------------------------------------------------"
                << std::endl;
      std::cerr << "Exception on processing: " << std::endl
                << exc.what() << std::endl
                << "Aborting!" << std::endl
                << "----------------------------------------------------"
                << std::endl;
 
      return 1;
    }
  catch (...) 
    {
      std::cerr << std::endl << std::endl
                << "----------------------------------------------------"
                << std::endl;
      std::cerr << "Unknown exception!" << std::endl
                << "Aborting!" << std::endl
                << "----------------------------------------------------"
                << std::endl;
      return 1;
    }
 
  return 0;
}

template class ParsedFunction<2>;
template class ParsedFunction<3>;
